%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Dynare set-up:
%   - NK Model with costs of adjusting investment
%   - Non-linear set-up
%   - With habit and indexing
%   - Remove retail costs
%   - Add Flexi Price Bloc
%   - Non-Zero ss inflation
%   - Use of preprocessor commands
%
% (c) CIMS Univeristy of Surrey
% The Science and  Art of DSGE Modelling:
% Construction, Calibration, Estimation and Policy
%
% This is a free software: you can redistribute it and/or modify it under
% the terms of the GNU General Public License as published by the Free
% Software Foundation, either version 3 of the License, or (at your option)
% any later version.  See <http://www.gnu.org/licenses/> for more information.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%----------------------------------------------------------------
% 0. Housekeeping (close all graphic windows)
%----------------------------------------------------------------
close all;

%----------------------------------------------------------------
% Control panel
%----------------------------------------------------------------

@#define habit = 1
   
@#define indexing = 1 

@#define rule = 0

%rule=0  gives  implementable rule with persistence, inflation and output targets
%rule=1  gives  standard Taylor  rule with persistence, inflation and output gap targets

@#define flexi = 1

%flexi=0 gives NK model with sticky prices
%flexi=1 makes prices flexi- should then give the same output as RBC

%----------------------------------------------------------------
% 1. Defining variables
%----------------------------------------------------------------
%   This section defines each variable and parameters using Latex 
%   syntax so to trigger Dynare producing a .tex file with all
%   the equilibrium conditions in the .mod file

%----------------------------------------------------------------
% Declaration of endogeneous variables
%----------------------------------------------------------------


var
%    U           $U$         (long_name='Utility')
%    UC          $UC$        (long_name='Marginal utility of consumption')
%    UH          $UH$        (long_name='Marginal utility of labour')
%    LAMBDA      $\lambda$   (long_name='Stochastic discount factor')
%    R           $R$         (long_name='Risk free interest rate')
%    RK          $R^{K}$     (long_name='Return on capital')
    C           $C$         (long_name='Consumption') 
    W           $W$         (long_name='Real wage')
    H           $H$         (long_name='Hours')
%    Y           $Y$         (long_name='Output') 
%    YW          $Y^{W}$     (long_name='Intermediate output')
%    PWP         $P^{W}$     (long_name='Intermediate price')
    K           $K$         (long_name='Capital stock')
    I           $I$         (long_name='Investment')  
%    tax         $Tax$       (long_name='Taxes') 
    A           $A$         (long_name='Production technology')
    G           $G$         (long_name='Government spending') 
%    KYW         $\frac{\bar{K}}{\bar{YW}}$   (long_name='Capital output ratio in steady state')
%    IY          $\frac{\bar{I}}{\bar{Y}}$    (long_name='Investment output ratio in steady state')
%    CY          $\frac{\bar{C}}{\bar{Y}}$    (long_name='Consumption output ratio in steady state')
%    S           $S$                          (long_name='Investment adjustment costs') 
%    Sdash       $S^{l}$                      (long_name='Adjustment costs on X')
    Q           $Q$                          (long_name='Tobin Q')
%    X           $X$                          (long_name='Gross growth rate of investment')
    Rn          $Rn$                         (long_name='Nominal interest rate') 
%    ER          $E(R)$                       (long_name='Ex-post real interest rate')
    PIE         $Pi$                         (long_name='Inflation')  
%    MC          $MC$                         (long_name='Real marginal costs')
    JJ          $JJ$                         (long_name='Price setting - Recursive auxiliary variable 1a')
%    JJtilde     $\tilde{JJ}$                 (long_name='Price setting - Recursive auxiliary variable 2a')
    J           $J$                          (long_name='Price setting - Recursive auxiliary variable 1b')
%    Jtilde      $\tilde{J}$                  (long_name='Price setting - Recursive auxiliary variable 2b')
%    PIEtilde    $\tilde{\pi}$                (long_name='Optimal reset price')            
    Delta       $\delta$                     (long_name='Price dispersion')
    MS          $MS$                         (long_name='Mark-up shock')

 %   UF           $UF$         (long_name='Utility - flexible prices')
 %   UCF          $UCF$        (long_name='Marginal utility of consumption - flexible prices')
 %   UHF          $ULF$        (long_name='Marginal utility of labour - flexible prices')    
 %   RF           $RF$         (long_name='Risk free interest rate - flexible prices')
 %   PWPF         $PF^{W}$     (long_name='Wholesale price - flxible prices') 
    HF           $HF$         (long_name='Hours - flxible prices')  
 %   YF           $YF$         (long_name='Output - flxible prices')  
    KF           $KF$         (long_name='Capital stock - flxible prices') 
    IF           $IF$         (long_name='Investment - flxible prices')   
    GF           $GF$         (long_name='Government spending - flexible prices')
    WF           $WF$         (long_name='Real wage - flxible prices') 
    CF           $CF$         (long_name='Consumption - flxible prices') 
 %   KYF          $\frac{\bar{KF}}{\bar{YF}}$   (long_name='Capital output ratio in steady state - flxible prices')
 %   IYF          $\frac{\bar{IF}}{\bar{YF}}$   (long_name='Investment output ratio in steady state - flxible prices') 
 %   CYF          $\frac{\bar{CF}}{\bar{YF}}$   (long_name='Consumption output ratio in steady state - flxible prices')
 %   taxF         $TaxF$       (long_name='Taxes - flxible prices')
 %   LAMBDAF      $\lambda$    (long_name='Stochastic discount factor - flxible prices')
 %   RKF          $RFk$        (long_name='Gross return on capital')
 %   SF           $SF$         (long_name='Investment adjustment costs')
 %   SdashF       $SF^{l}$     (long_name='Adjustment costs on X')
    QF           $QF$         (long_name='Tobin Q') 
 %   XF           $XF$         (long_name='Gross growth rate of investment')
 
    OUTGAP       $OUTGAP$     (long_name='Output gap')  

%--------------------------------------------------------------------------------
%   Variables in deviation from the steady state (used for IRFs)
%--------------------------------------------------------------------------------

    RR              $RR$       (long_name='Real interest rate - deviation from the steady state') 
    RnRn            $RRn$      (long_name='Nominal interest rate - deviation from the steady state') 
    ERER            $E(RR)$    (long_name='Ex-post real interest rate - deviation from the steady state')
    YY              $yy$       (long_name='Output - deviation from the steady state')
    CC              $cc$       (long_name='Consumption - deviation from the steady state') 
    HH              $hh$       (long_name='Hours - deviation from the steady state') 
    WW              $ww$       (long_name='Real wage - deviation from the steady state')
    II              $ii$       (long_name='Investment - deviation from the steady state') 
    KK              $kk$       (long_name='Capital - deviation from the steady state') 
    QQ              $qq$       (long_name='Tobin Q - deviation from the steady state')  
    PIEPIE          $\pi\pi$   (long_name='Inflation - deviation from the steady state')
    OUTGAPOUTGAP    $Ogapogap$ (long_name='Output gap - deviation from the steady state')
;


%----------------------------------------------------------------
% Declaration of exogeneous variables
%----------------------------------------------------------------

varexo
    epsA        ${\epsilon^{A}}$      (long_name='Labor augmenting shock')
    epsG        ${\epsilon^{G}}$      (long_name='Government spending shock')
    epsMS       ${\epsilon^{MS}}$     (long_name='Mark-up shock')
    epsM        ${\epsilon^{M}}$      (long_name='Monetary policy shock')
;

%----------------------------------------------------------------
% Declaration of parameters
%----------------------------------------------------------------

parameters
    cy             ${cy}$                       (long_name='Consumption output ratio in steady state')
    iy             ${iy}$                       (long_name='Investment output ratio in steady state') 
    gy             ${gy}$                       (long_name='Government spending output ratio in steady state') 
    zzeta          ${\zeta}$                    (long_name='Elasticity of substitution across products')
    alp            ${\alpha}$                   (long_name='Labour share')
    betta          ${\beta}$                    (long_name='Discount factor') 
    delta          ${\delta}$                   (long_name='Capital depreciation')
    sigma_c        ${\sigma_{C}}$               (long_name='Inverse of the elasticity of substitution')
    varrho         ${\varrho}$                  (long_name='Weight on Leisure in utility') 
    rhoA           ${\rho_{A}}$                 (long_name='Persistence of labor augmentig shock')
    rhoG           ${\rho_{G}}$                 (long_name='Persistence of government spending shock')
    A_bar          ${\bar{A}}$                  (long_name='Labor technology in steady state')
    phiX           ${\phi_{X}}$                 (long_name='Smoothing parameter of investment adjustment costs')
    xi             ${\xi}$                      (long_name='Calvo parameter')          
    rho_r          ${\rho_{R}}$                 (long_name='Interest rate smoothing parameter') 
    theta_pie      ${\theta^{\pi}$              (long_name='Taylor rule parameter for inflation')
    theta_y        ${\theta^{y}$                (long_name='Taylor rule parameter for output') 
    rhoMS          ${\rho_{MS}}$                (long_name='Persistence of mark-up shock')
    chi            ${\chi}$                     (long_name='habits persistence')
    gammap         ${\gamma^{p}}$               (long_name='degree of indexation')
    H_bar          ${\bar{H}}$                  (long_name='Hours in steady state') 
    PIE_bar        ${\bar{\pi}}$                (long_name='Inflation in steady state')
    sigma_A        ${\sigma_{A}}$               (long_name='STD of labor augmentig shock')
    sigma_G        ${\sigma_{G}}$               (long_name='STD of government spendng shock') 
    sigma_MS       ${\sigma_{MS}}$              (long_name='STD of mark-up shock') 
    sigma_M        ${\sigma_{M}}$               (long_name='STD of monetary policy shock')
;


%----------------------------------------------------------------
% 2. Calibration
%----------------------------------------------------------------

PIE_bar=1.01;       // Steady state inflation
gammap=0.5;         // Indexing
cy=0.6;             // Consumption output ratio in steady state
gy=0.2;             // Government output ratio
iy=0.2;             // Investment output ratio in steady state
chi=0.7;            // Habit parameter
H_bar=0.35;         // Steady state hours
phiX=2.0;           // Smoothing parameter of investment adjustment costs
alp=0.70;           // Labour share
betta=0.99;         // Discount factor
sigma_c=2.0;        // Inverse of the elasticity of substitution

@#if habit > 0
    chi=0.7;
   @#else
   chi=0;
@#endif

@#if indexing > 0
     gammap=0.5;
    @#else
    gammap=0;
@#endif
varrho=alp*(1-H_bar)/(alp*(1-H_bar)+cy*(1-chi)*H_bar); // Capital depreciation
                                                       // Calibrated to hit H=H_bar in steady state
delta=(1/betta-1)*iy/(1-alp-iy);                       // Capital depreciation
                                                       // Calibrated to hit iy and R=1/betta in steady state

%----------------------------------------------------------------
% Monetary Policy Rule
%----------------------------------------------------------------

rho_r=0.7;     // Smoothing parameter of the interest rate
theta_pie=1.5; // Taylor rule parameter for inflation
theta_y=0.3;   // Taylor rule parameter for output
%----------------------------------------------------------------
% Choice of units of output, hours and capital (i.e,. normalization)
%----------------------------------------------------------------
A_bar=1;            // Steady state technology            

%----------------------------------------------------------------
% Shock persistence
%----------------------------------------------------------------

rhoA=0.75; // Persistence of labor augmentig shock
rhoG=0.75; // Persistence of government spending shock
rhoMS=0.75;// Persistence of mark-up shock

%----------------------------------------------------------------
% Standard deviation of shocks
%----------------------------------------------------------------
sigma_A=0.01; // STD of labor augmentig shock
sigma_G=0.01; // STD of government spendng shock
sigma_MS=0.01;// STD of mark-up shock
sigma_M=0.001;// STD of monetary policy shock

%-----------------------------------------------------------------

@#if flexi==0
    xi=0.75;
%    xi=0.5; 
    zzeta=7;
@#else

% flexi-price case 
xi=0.0;
zzeta=1000;
//sigma_MS=0;//ensures NK and RBC are equivalent

@#endif
%-------------------------------------------------------------------

%----------------------------------------------------------------
% 3. Model
%----------------------------------------------------------------

model;

%    [name='Single period utility']

#U=((((C-chi*C(-1))^(1-varrho))*((1-H)^varrho))^(1-sigma_c)-1)/(1-sigma_c);

%    [name='Marginal Utility of Consumption (with external habits)']
    
#UC=(1-varrho)*((C-chi*C(-1))^((1-varrho)*(1-sigma_c)-1))*((1-H)^(varrho*(1-sigma_c)));
#UC_LAG=(1-varrho)*((C(-1)-chi*C(-2))^((1-varrho)*(1-sigma_c)-1))*((1-H(-1))^(varrho*(1-sigma_c)));
#UC_LEAD=(1-varrho)*((C(+1)-chi*C)^((1-varrho)*(1-sigma_c)-1))*((1-H(+1))^(varrho*(1-sigma_c)));

%    [name='Marginal Utility of Labour (with external habits)']
    
#UH=-varrho*((C-chi*C(-1))^((1-varrho)*(1-sigma_c)))*((1-H)^(varrho*(1-sigma_c)-1));

%    [name='Stochastic discount factor - 1']

#LAMBDA=betta*UC/UC_LAG;
#LAMBDA_LEAD=betta*UC_LEAD/UC;

%    [name='Gross growth rate of investment']

#X = I/I(-1);
#X_LEAD = I(+1)/I;

%    [name='Investment adjustment cost - 1']

#S=phiX*(X-1)^2;

%    [name='Investment adjustment cost - 2']

#Sdash=2*phiX*(X-1);
#Sdash_LEAD=2*phiX*(X_LEAD-1);

%    [name='Fisher equation - real interest rate']    
       
#R=Rn(-1)/PIE;
#R_LEAD=Rn/PIE(+1);    

%    [name='Fisher equation - expected real interest rate']        
    
#ER=Rn/PIE(+1);


    %----------------------------------------------------------------
    % Euler equation 
    %----------------------------------------------------------------
    %Risky Bonds; Rn set in period t to yield interest over period t+1, 
    %ex post real interest rate R now risky

        %UC=betta*R_LEAD*UC_LEAD;

    %----------------------------------------------------------------
    % Stochastic discount factor
    %----------------------------------------------------------------


    %Alternative Euler Equation

%    [name='Stochastic discount factor - 2']

        R_LEAD*LAMBDA_LEAD=1;

    %For NK: Fisher Equation

%    [name='Labor Supply FOC']

    W=-UH/UC;

%    [name='Prodution function']

    %Note: Capital stock is end-of-period

#YW_LAG=(((A(-1)*H(-1))^alp)*(K(-2))^(1-alp));    
#YW=(((A*H)^alp)*(K(-1))^(1-alp));
#YW_LEAD=(((A(+1)*H(+1))^alp)*(K)^(1-alp));


%     [name='Ratio between prodution and price dispertion']

#Y_LAG=YW_LAG/Delta(-1);
#Y=YW/Delta;

%    [name='Wholesale firms FOC - 3']

 #PWP=W*H/(alp*YW);
 #PWP_LEAD=W(+1)*H(+1)/(alp*YW_LEAD);

%    [name='Mark-up Monopolistic pricing']

#MC=PWP;

%    [name='Inflation dynamic - 6']

#PIEtilde=PIE/PIE(-1)^gammap;
#PIEtilde_LEAD=PIE(+1)/PIE^gammap;

%    [name='Inflation dynamic - 3']

#JJtilde=(PIEtilde^(zzeta-1))*JJ;
#JJtilde_LEAD=(PIEtilde_LEAD^(zzeta-1))*JJ(+1);


%    [name='Inflation dynamic - 4']

#Jtilde=PIEtilde^zzeta*J;
#Jtilde_LEAD=PIEtilde_LEAD^zzeta*J;

%    [name='Inflation dynamic - 1']

        JJ=Y*UC+xi*betta*JJtilde_LEAD;

%    [name='Inflation dynamic - 2']

        J=(1/(1-(1/zzeta)))*Y*UC*MC*MS+xi*betta*Jtilde_LEAD;

%    [name='Inflation dynamic - 5']

        1=xi*(PIEtilde^(zzeta-1))+(1-xi)*((J/JJ)^(1-zzeta));

%    [name='Price dispertion - eq. 10']    

    Delta=xi*(PIEtilde^zzeta)*Delta(-1)+(1-xi)*(J/JJ)^(-zzeta);


%    [name='Wholesale firms FOC - 1a']
    
 #RK=((1-alp)*PWP*YW/K(-1)+(1-delta)*Q)/Q(-1);
 #RK_LEAD=((1-alp)*PWP_LEAD*YW_LEAD/K+(1-delta)*Q(+1))/Q;


%    [name='Wholesale firms FOC - 1b']
    %where Tobin's Q is given by

        Q*(1-S-X*Sdash)+LAMBDA_LEAD*Q(+1)*Sdash_LEAD*X_LEAD^2=1;

%    [name='Wholesale firms FOC - 2']
 
        LAMBDA_LEAD*RK_LEAD=1;
    
%    [name='Resource constraint']
   
    Y=G+C+I;

%    [name='Balanced budget constraint']

#tax=G;

%    [name='Capital law of motion']

        K=(1-S)*I+(1-delta)*K(-1);

    @#if rule==0

%    [name='Taylor rule - implementable rule (relative to ss)']

        log(Rn/STEADY_STATE(Rn))=rho_r*log(Rn(-1)/STEADY_STATE(Rn))+(1-rho_r)*(theta_pie*log(PIE/STEADY_STATE(PIE))+theta_y*log(Y/STEADY_STATE(Y)))
        +sigma_M*epsM;
    @#endif

    @#if rule>0
%   [name='Taylor rule - conventional  rule (relative to Flexi-Price )']
        
    log(Rn/STEADY_STATE(Rn))=rho_r*log(Rn(-1)/STEADY_STATE(Rn))+(1-rho_r)*(theta_pie*log(PIE/STEADY_STATE(PIE))
    +theta_y*log((Y/STEADY_STATE(Y))/(YF/STEADY_STATE(YF))))+epsM;
    @#endif

%    [name='Labour augmenting shock process']

        log(A)-log(STEADY_STATE(A))=rhoA*(log(A(-1))-log(STEADY_STATE(A)))+sigma_A*epsA;

%    [name='Government shock process']

        log(G)-log(STEADY_STATE(G))=rhoG*(log(G(-1))-log(STEADY_STATE(G)))+sigma_G*epsG;

    [name='Government shock process - flexible prices']

        log(GF)-log(STEADY_STATE(GF))=rhoG*(log(GF(-1))-log(STEADY_STATE(GF)))+sigma_G*epsG;

%    [name='Markup shock']

        log(MS)-log(STEADY_STATE(MS))=rhoMS*(log(MS(-1))-log(STEADY_STATE(MS)))+sigma_MS*epsMS;


%----------------------------------------------------------------
% Ratio variables used in steady state
%----------------------------------------------------------------

        #KYW=K/YW;
        #IY=I/Y;
        #CY=C/Y;


%---------------------------------------------------------------------------
%---------------------------------------------------------------------------
%
% FLEXI- PRICE BLOCK
%
%---------------------------------------------------------------------------
%---------------------------------------------------------------------------


%    [name='Single period utility']

#UF=((((CF-chi*CF(-1))^(1-varrho))*((1-HF)^varrho))^(1-sigma_c)-1)/(1-sigma_c);

%    [name='Marginal Utility of Consumption (with external habits)']

#UCF=(1-varrho)*((CF-chi*CF(-1))^((1-varrho)*(1-sigma_c)-1))*((1-HF)^(varrho*(1-sigma_c)));
#UCF_LAG=(1-varrho)*((CF(-1)-chi*CF(-2))^((1-varrho)*(1-sigma_c)-1))*((1-HF(-1))^(varrho*(1-sigma_c)));
#UCF_LEAD=(1-varrho)*((CF(+1)-chi*CF)^((1-varrho)*(1-sigma_c)-1))*((1-HF(+1))^(varrho*(1-sigma_c)));

%    [name='Marginal Utility of Labour (with external habits)']

#UHF=-varrho*((CF-chi*CF(-1))^((1-varrho)*(1-sigma_c)))*((1-HF)^(varrho*(1-sigma_c)-1));

%    [name='Stochastic Discount Factor - 1']

#LAMBDAF=betta*UCF/UCF_LAG;
#LAMBDAF_LEAD=betta*UCF_LEAD/UCF;

%    [name='Gross growth rate of investment']

#XF=IF/IF(-1);
#XF_LEAD=IF(+1)/IF;

%    [name='Investment adjustment cost - 1']

#SF=phiX*(XF-1)^2;

%    [name='Investment adjustment cost - 2']

#SdashF=2*phiX*(XF-1);
#SdashF_LEAD=2*phiX*(XF_LEAD-1);


    %-------------------------------------------------------------------------
    % Euler equation
    %-------------------------------------------------------------------------
    %Riskless Bonds; R set in period t to yield interest over period t+1

        %UCF=betta*RF*UCF(+1);

    %-------------------------------------------------------------------------
    % Stochastic Discount Factor
    %-------------------------------------------------------------------------

    %Alternative Euler Equation

%    [name='Stochastic Discount Factor - 2']

#RF=1/LAMBDAF_LEAD;

%     [name='Labour supply foc - eq 54']

        -UHF/UCF=WF;

%    [name='Prodution function']
    %Note: Capital stock is end-of-period

#YF_LAG=((A(-1)*HF(-1))^alp)*(KF(-2))^(1-alp);
#YF=((A*HF)^alp)*(KF(-1))^(1-alp);
#YF_LEAD=((A(+1)*HF(+1))^alp)*(KF)^(1-alp);

%    [name='Wholesale firms FOC - 3']

#PWPF=WF*HF/(alp*YF);
#PWPF_LEAD=WF(+1)*HF(+1)/(alp*YF_LEAD);


%    [name='Wholesale firms FOC - 1a']

#RKF=((1-alp)*PWPF*YF/KF(-1)+(1-delta)*QF)/QF(-1);
#RKF_LEAD=((1-alp)*PWPF_LEAD*YF_LEAD/KF+(1-delta)*QF(+1))/QF;


%    [name='Wholesale firms FOC - 2']

        LAMBDAF_LEAD*RKF_LEAD=1;

%    [name='Wholesale firms FOC - 1b']
        %where Tobin's Q is given by

        QF*(1-SF-XF*SdashF)+LAMBDAF_LEAD*QF(+1)*SdashF_LEAD*XF_LEAD^2=1;


%    [name='Resource constraint - eq 58']

        YF=CF+GF+IF;

%    [name='Capital law of motion']

        KF=(1-SF)*IF+(1-delta)*KF(-1); 
    %where

%    [name='Balance budget constraint']

#taxF=GF;

%    [name='Mark-up monopolitstic shock']

        PWPF=(1-1/zzeta)/MS;

%    [name='Output gap']

OUTGAP=YF/Y;

%----------------------------------------------------------------
% Variables used for the steady state
%----------------------------------------------------------------
#KYF=KF/YF;
#IYF=IF/YF;
#CYF=CF/YF;

%----------------------------------------------------------------
% Variables in deviation form for IRFs
%----------------------------------------------------------------

YY=Y/STEADY_STATE(Y);
KK=K/STEADY_STATE(K);
II=I/STEADY_STATE(I);
CC=C/STEADY_STATE(C);
WW=W/STEADY_STATE(W);
HH=H/STEADY_STATE(H);
QQ=Q/STEADY_STATE(Q);
RR=R/STEADY_STATE(R);
ERER=ER/STEADY_STATE(ER);
RnRn=Rn/STEADY_STATE(Rn);
PIEPIE=PIE/STEADY_STATE(PIE);
OUTGAPOUTGAP=OUTGAP/STEADY_STATE(OUTGAP);
end;

//initval;
//end;


%----------------------------------------------------------------
% 4. Computation
%----------------------------------------------------------------

%----------------------------------------------------------------
% Analytical steady-state computation - see Section 2.4 of notes
%----------------------------------------------------------------

steady_state_model;
    H=H_bar;
    A=A_bar;
    PIE=PIE_bar;
%    R=1.0/betta;
    //PWP=1-1/zzeta;// and
    //MC=PWP;
    //Delta=1;// for PIE=1 replaced with
%    PIEtilde=PIE^(1-gammap);
%    MC=(1-1/zzeta)*(1-xi*betta*(PIE^(1-gammap))^zzeta)/(1-xi*betta*(PIE^(1-gammap))^(zzeta-1))*(((1-xi*(PIE^(1-gammap))^(zzeta-1))/(1-xi))^(1/(1-zzeta)));
%   PWP=MC;
    Delta=(1-xi)^(1/(1-zzeta))*(1-xi*(PIE^(1-gammap))^(zzeta-1))^(-zzeta/(1-zzeta))/(1-xi*(PIE^(1-gammap))^zzeta);
%KYW=((1-1/zzeta)*(1-xi*betta*(PIE^(1-gammap))^zzeta)/(1-xi*betta*(PIE^(1-gammap))^(zzeta-1))*(((1-xi*(PIE^(1-gammap))^(zzeta-1))/(1-xi))^(1/(1-zzeta))))*(1-alp)/((1.0/betta)-1+delta);
%IY=Delta*delta*(((1-1/zzeta)*(1-xi*betta*(PIE^(1-gammap))^zzeta)/(1-xi*betta*(PIE^(1-gammap))^(zzeta-1))*(((1-xi*(PIE^(1-gammap))^(zzeta-1))/(1-xi))^(1/(1-zzeta))))*(1-alp)/((1.0/betta)-1+delta));
%CY=1-(Delta*delta*(((1-1/zzeta)*(1-xi*betta*(PIE^(1-gammap))^zzeta)/(1-xi*betta*(PIE^(1-gammap))^(zzeta-1))*(((1-xi*(PIE^(1-gammap))^(zzeta-1))/(1-xi))^(1/(1-zzeta))))*(1-alp)/((1.0/betta)-1+delta)))-gy;
    //H=(1-varrho)*alp*((1-1/zzeta)*(1-xi*betta*(PIE^(1-gammap))^zzeta)/(1-xi*betta*(PIE^(1-gammap))^(zzeta-1))*(((1-xi*(PIE^(1-gammap))^(zzeta-1))/(1-xi))^(1/(1-zzeta))))/(varrho*(1-(Delta*delta*(((1-1/zzeta)*(1-xi*betta*(PIE^(1-gammap))^zzeta)/(1-xi*betta*(PIE^(1-gammap))^(zzeta-1))*(((1-xi*(PIE^(1-gammap))^(zzeta-1))/(1-xi))^(1/(1-zzeta))))*(1-alp)/((1.0/betta)-1+delta)))-gy)*(1-chi)+(1-varrho)*alp*((1-1/zzeta)*(1-xi*betta*(PIE^(1-gammap))^zzeta)/(1-xi*betta*(PIE^(1-gammap))^(zzeta-1))*(((1-xi*(PIE^(1-gammap))^(zzeta-1))/(1-xi))^(1/(1-zzeta))))*Delta)//if varrho not calibrated;
    varrho=(1-H)/(1+H*((1-(Delta*delta*(((1-1/zzeta)*(1-xi*betta*(PIE^(1-gammap))^zzeta)/(1-xi*betta*(PIE^(1-gammap))^(zzeta-1))*(((1-xi*(PIE^(1-gammap))^(zzeta-1))/(1-xi))^(1/(1-zzeta))))*(1-alp)/((1.0/betta)-1+delta)))-gy)*(1-chi)/(alp*((1-1/zzeta)*(1-xi*betta*(PIE^(1-gammap))^zzeta)/(1-xi*betta*(PIE^(1-gammap))^(zzeta-1))*(((1-xi*(PIE^(1-gammap))^(zzeta-1))/(1-xi))^(1/(1-zzeta))))*Delta)-1));
%    YW=A*H*(((1-1/zzeta)*(1-xi*betta*(PIE^(1-gammap))^zzeta)/(1-xi*betta*(PIE^(1-gammap))^(zzeta-1))*(((1-xi*(PIE^(1-gammap))^(zzeta-1))/(1-xi))^(1/(1-zzeta))))*(1-alp)/((1.0/betta)-1+delta))^((1-alp)/alp);
%    Y=(A*H*(((1-1/zzeta)*(1-xi*betta*(PIE^(1-gammap))^zzeta)/(1-xi*betta*(PIE^(1-gammap))^(zzeta-1))*(((1-xi*(PIE^(1-gammap))^(zzeta-1))/(1-xi))^(1/(1-zzeta))))*(1-alp)/((1.0/betta)-1+delta))^((1-alp)/alp))/Delta;
    K=(((1-1/zzeta)*(1-xi*betta*(PIE^(1-gammap))^zzeta)/(1-xi*betta*(PIE^(1-gammap))^(zzeta-1))*(((1-xi*(PIE^(1-gammap))^(zzeta-1))/(1-xi))^(1/(1-zzeta))))*(1-alp)/((1.0/betta)-1+delta))*(A*H*(((1-1/zzeta)*(1-xi*betta*(PIE^(1-gammap))^zzeta)/(1-xi*betta*(PIE^(1-gammap))^(zzeta-1))*(((1-xi*(PIE^(1-gammap))^(zzeta-1))/(1-xi))^(1/(1-zzeta))))*(1-alp)/((1.0/betta)-1+delta))^((1-alp)/alp));
    I=(Delta*delta*(((1-1/zzeta)*(1-xi*betta*(PIE^(1-gammap))^zzeta)/(1-xi*betta*(PIE^(1-gammap))^(zzeta-1))*(((1-xi*(PIE^(1-gammap))^(zzeta-1))/(1-xi))^(1/(1-zzeta))))*(1-alp)/((1.0/betta)-1+delta)))*((A*H*(((1-1/zzeta)*(1-xi*betta*(PIE^(1-gammap))^zzeta)/(1-xi*betta*(PIE^(1-gammap))^(zzeta-1))*(((1-xi*(PIE^(1-gammap))^(zzeta-1))/(1-xi))^(1/(1-zzeta))))*(1-alp)/((1.0/betta)-1+delta))^((1-alp)/alp))/Delta);
    G=gy*((A*H*(((1-1/zzeta)*(1-xi*betta*(PIE^(1-gammap))^zzeta)/(1-xi*betta*(PIE^(1-gammap))^(zzeta-1))*(((1-xi*(PIE^(1-gammap))^(zzeta-1))/(1-xi))^(1/(1-zzeta))))*(1-alp)/((1.0/betta)-1+delta))^((1-alp)/alp))/Delta);
    W=alp*((1-1/zzeta)*(1-xi*betta*(PIE^(1-gammap))^zzeta)/(1-xi*betta*(PIE^(1-gammap))^(zzeta-1))*(((1-xi*(PIE^(1-gammap))^(zzeta-1))/(1-xi))^(1/(1-zzeta))))*(A*H*(((1-1/zzeta)*(1-xi*betta*(PIE^(1-gammap))^zzeta)/(1-xi*betta*(PIE^(1-gammap))^(zzeta-1))*(((1-xi*(PIE^(1-gammap))^(zzeta-1))/(1-xi))^(1/(1-zzeta))))*(1-alp)/((1.0/betta)-1+delta))^((1-alp)/alp))/H;
    C=(1-(Delta*delta*(((1-1/zzeta)*(1-xi*betta*(PIE^(1-gammap))^zzeta)/(1-xi*betta*(PIE^(1-gammap))^(zzeta-1))*(((1-xi*(PIE^(1-gammap))^(zzeta-1))/(1-xi))^(1/(1-zzeta))))*(1-alp)/((1.0/betta)-1+delta)))-gy)*((A*H*(((1-1/zzeta)*(1-xi*betta*(PIE^(1-gammap))^zzeta)/(1-xi*betta*(PIE^(1-gammap))^(zzeta-1))*(((1-xi*(PIE^(1-gammap))^(zzeta-1))/(1-xi))^(1/(1-zzeta))))*(1-alp)/((1.0/betta)-1+delta))^((1-alp)/alp))/Delta);

%   Other endogenous Variabes

%    tax=G;
%    U=1/(1-sigma_c)*((C*(1-chi))^((1-varrho)*(1-sigma_c))*(1-H)^(varrho*(1-sigma_c))-1);
%   UC=(1-varrho)*(C*(1-chi))^((1-varrho)*(1-sigma_c)-1)*(1-H)^(varrho*(1-sigma_c));
%    UH=-varrho*((C*(1-chi))^((1-varrho)*(1-sigma_c)))*((1-H)^(varrho*(1-sigma_c)-1));
%    LAMBDA=betta;
%     RK=(1.0/betta);

%    S=0;
%    Sdash=0;
    Q=1;
   % X=1;

   JJ=((A*H*(((1-1/zzeta)*(1-xi*betta*(PIE^(1-gammap))^zzeta)/(1-xi*betta*(PIE^(1-gammap))^(zzeta-1))*(((1-xi*(PIE^(1-gammap))^(zzeta-1))/(1-xi))^(1/(1-zzeta))))*(1-alp)/((1.0/betta)-1+delta))^((1-alp)/alp))/Delta)*((1-varrho)*(C*(1-chi))^((1-varrho)*(1-sigma_c)-1)*(1-H)^(varrho*(1-sigma_c)))/(1-betta*xi*(PIE^(1-gammap))^(zzeta-1));
%   JJtilde=((PIE^(1-gammap))^(zzeta-1))*JJ;
    J=(1/(1-1/zzeta))*((A*H*(((1-1/zzeta)*(1-xi*betta*(PIE^(1-gammap))^zzeta)/(1-xi*betta*(PIE^(1-gammap))^(zzeta-1))*(((1-xi*(PIE^(1-gammap))^(zzeta-1))/(1-xi))^(1/(1-zzeta))))*(1-alp)/((1.0/betta)-1+delta))^((1-alp)/alp))/Delta)*((1-varrho)*(C*(1-chi))^((1-varrho)*(1-sigma_c)-1)*(1-H)^(varrho*(1-sigma_c)))*((1-1/zzeta)*(1-xi*betta*(PIE^(1-gammap))^zzeta)/(1-xi*betta*(PIE^(1-gammap))^(zzeta-1))*(((1-xi*(PIE^(1-gammap))^(zzeta-1))/(1-xi))^(1/(1-zzeta))))/(1-betta*xi*(PIE^(1-gammap))^(zzeta));
%   Jtilde=(PIE^(1-gammap))^zzeta*J;

    MS=1;
    Rn=PIE/betta;
%    ER=Rn/PIE;

    YY=1;
    CC=1;
    HH=1;
    WW=1; 
    II=1; 
    KK=1;
    RR=1;
    QQ=1;
    RnRn=1;
    PIEPIE=1;
    ERER=1;
    
%RF=(1.0/betta);
%PWPF=1-1/zzeta;
%KYF=(1-1/zzeta)*(1-alp)/((1.0/betta)-1+delta);
%IYF=(1-1/zzeta)*(1-alp)*delta/((1.0/betta)-1+delta);
%CYF=1-((1-1/zzeta)*(1-alp)*delta/((1.0/betta)-1+delta))-gy;
    HF=(1-varrho)*alp*(1-1/zzeta)/(varrho*(1-((1-1/zzeta)*(1-alp)*delta/((1.0/betta)-1+delta))-gy)*(1-chi)+(1-varrho)*alp*(1-1/zzeta));
%YF=A*HF*((1-1/zzeta)*(1-alp)/((1.0/betta)-1+delta))^((1-alp)/alp);
    GF=gy*(A*HF*((1-1/zzeta)*(1-alp)/((1.0/betta)-1+delta))^((1-alp)/alp));
    KF=((1-1/zzeta)*(1-alp)/((1.0/betta)-1+delta))*(A*HF*((1-1/zzeta)*(1-alp)/((1.0/betta)-1+delta))^((1-alp)/alp));
    IF=((1-1/zzeta)*(1-alp)*delta/((1.0/betta)-1+delta))*(A*HF*((1-1/zzeta)*(1-alp)/((1.0/betta)-1+delta))^((1-alp)/alp));
    WF=alp*(1-1/zzeta)*(A*HF*((1-1/zzeta)*(1-alp)/((1.0/betta)-1+delta))^((1-alp)/alp))/HF;
    CF=(1-((1-1/zzeta)*(1-alp)*delta/((1.0/betta)-1+delta))-gy)*(A*HF*((1-1/zzeta)*(1-alp)/((1.0/betta)-1+delta))^((1-alp)/alp));

 %taxF=GF;
 %UF=1/(1-sigma_c)*((CF*(1-chi))^((1-varrho)*(1-sigma_c))*(1-HF)^(varrho*(1-sigma_c))-1);
 %UCF=(1-varrho)*(CF*(1-chi))^((1-varrho)*(1-sigma_c)-1)*(1-HF)^(varrho*(1-sigma_c));
 %UHF=-varrho*((CF*(1-chi))^((1-varrho)*(1-sigma_c)))*((1-HF)^(varrho*(1-sigma_c)-1));
 %LAMBDAF=betta;
 %RKF=(1.0/betta);
 %SF=0;%S;
 %   SdashF=0;%Sdash;
    QF=Q;
  %XF=1;%X;
  OUTGAP=(A*HF*((1-1/zzeta)*(1-alp)/((1.0/betta)-1+delta))^((1-alp)/alp))/((A*H*(((1-1/zzeta)*(1-xi*betta*(PIE^(1-gammap))^zzeta)/(1-xi*betta*(PIE^(1-gammap))^(zzeta-1))*(((1-xi*(PIE^(1-gammap))^(zzeta-1))/(1-xi))^(1/(1-zzeta))))*(1-alp)/((1.0/betta)-1+delta))^((1-alp)/alp))/Delta);
    OUTGAPOUTGAP=1;
end;
steady;

%----------------------------------------------------------------
% Specification of shocks
%----------------------------------------------------------------

shocks;
    var epsA; stderr sigma_A;
    var epsG; stderr sigma_G;
    var epsMS; stderr sigma_MS;
    var epsM; stderr sigma_M;
end;

check;

stoch_simul(order=1,irf=40) YY CC II HH WW RR ERER QQ RnRn PIEPIE OUTGAPOUTGAP;

write_latex_dynamic_model;
write_latex_static_model;
write_latex_definitions;
write_latex_parameter_table;
collect_latex_files;
